<?php

declare(strict_types = 1);

error_reporting(E_ALL);

if(file_exists('liveproto.php') === false){
	copy('https://installer.liveproto.dev/liveproto.php','liveproto.php');
	require_once 'liveproto.php';
} else {
	require_once 'liveproto.phar';
}

use Tak\Liveproto\Network\Client;
use Tak\Liveproto\Utils\Settings;
use Tak\Liveproto\Filters\Filter;
use Tak\Liveproto\Filters\Filter\Command;
use Tak\Liveproto\Filters\Events\NewMessage;
use Tak\Liveproto\Filters\Interfaces\Incoming;
use Tak\Liveproto\Filters\Interfaces\IsPrivate;
use Tak\Liveproto\Enums\CommandType;
use Revolt\EventLoop;

// بارگذاری تنظیمات از config.php در صورت وجود
$config = [];
if (file_exists('config.php')) {
	$config = require 'config.php';
}
$GLOBALS['config'] = $config;

// تنظیمات کلاینت
$settings = new Settings();
$settings->setApiId($config['telegram']['api_id'] ?? 29784714);
$settings->setApiHash($config['telegram']['api_hash'] ?? '143dfc3c92049c32fbc553de2e5fb8e4');
$settings->setDeviceModel($config['telegram']['device_model'] ?? 'PC 64bit');
$settings->setSystemVersion($config['telegram']['system_version'] ?? '4.14.186');
$settings->setAppVersion($config['telegram']['app_version'] ?? '1.28.5');
$settings->setIPv6($config['telegram']['use_ipv6'] ?? false);
$settings->setHideLog($config['telegram']['hide_logs'] ?? true);
$settings->setReceiveUpdates($config['telegram']['receive_updates'] ?? false);

$client = new Client('ultra_gift_bot','sqlite',$settings);

// فایل برای ذخیره تنظیمات کاربران
const USERS_FILE = 'data/users.json';
const CONFIG_FILE = 'data/config.json';

// کلاس کمکی برای مدیریت داده‌ها
class DataManager {
    public static function loadUsers(): array {
        if (!file_exists(USERS_FILE)) {
            return [];
        }
        $data = file_get_contents(USERS_FILE);
        return json_decode($data, true) ?: [];
    }
    
    public static function saveUsers(array $users): void {
        if (!is_dir(dirname(USERS_FILE))) {
            @mkdir(dirname(USERS_FILE), 0755, true);
        }
        file_put_contents(USERS_FILE, json_encode($users, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    }
    
    public static function loadConfig(): array {
        $default = $GLOBALS['config']['monitoring'] ?? [
            'check_interval' => 5.0,
            'notification_enabled' => true,
            'show_limited_only' => true,
            'show_available_only' => true
        ];
        
        if (!file_exists(CONFIG_FILE)) {
            // ایجاد مسیر دایرکتوری در صورت نبود
            if (!is_dir(dirname(CONFIG_FILE))) {
                @mkdir(dirname(CONFIG_FILE), 0755, true);
            }
            file_put_contents(CONFIG_FILE, json_encode($default, JSON_PRETTY_PRINT));
            return $default;
        }
        
        $data = file_get_contents(CONFIG_FILE);
        return json_decode($data, true) ?: $default;
    }
    
    public static function saveConfig(array $config): void {
        if (!is_dir(dirname(CONFIG_FILE))) {
            @mkdir(dirname(CONFIG_FILE), 0755, true);
        }
        file_put_contents(CONFIG_FILE, json_encode($config, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    }
}

// فانکشن کمکی برای فرمت کردن پیام‌ها
function formatMessage(array $lines, mixed ...$values): string {
    return sprintf(implode(PHP_EOL, $lines), ...$values);
}

// فانکشن برای فرمت کردن تاریخ فارسی
function persianDate(): string {
    $months = [
        1 => 'فروردین', 2 => 'اردیبهشت', 3 => 'خرداد',
        4 => 'تیر', 5 => 'مرداد', 6 => 'شهریور',
        7 => 'مهر', 8 => 'آبان', 9 => 'آذر',
        10 => 'دی', 11 => 'بهمن', 12 => 'اسفند'
    ];
    
    $gregorian = date('Y-m-d');
    $jDate = new DateTime($gregorian);
    $year = $jDate->format('Y') - 621;
    $month = (int)$jDate->format('m');
    $day = $jDate->format('d');
    
    return sprintf('%d %s %d', $day, $months[$month], $year);
}

// کلاس اصلی هندلر
final class UltraGiftHandler {
    public array $users = [];
    public array $config = [];
    public array $giftStats = ['total_sent' => 0, 'unique_gifts' => 0];
    private array $sentGifts = []; // برای جلوگیری از ارسال مکرر
    
    public function __construct() {
        $this->users = DataManager::loadUsers();
        $this->config = DataManager::loadConfig();
        
        // بارگذاری آمار
        if (file_exists('data/stats.json')) {
            $stats = json_decode(file_get_contents('data/stats.json'), true);
            $this->giftStats = $stats ?: $this->giftStats;
        }
    }
    
    private function saveData(): void {
        DataManager::saveUsers($this->users);
        DataManager::saveConfig($this->config);
        if (!is_dir('data')) {
            @mkdir('data', 0755, true);
        }
        file_put_contents('data/stats.json', json_encode($this->giftStats, JSON_PRETTY_PRINT));
    }
    
    #[Filter(new NewMessage(new Command(start: [CommandType::SLASH, CommandType::DOT, CommandType::EXCLAMATION])))]
    public function start($update): void {
        $welcomeMsg = formatMessage([
            '🎁✨ سلام و خوش آمدید! ✨🎁',
            '',
            '🤖 به ربات هوشمند اطلاع‌رسانی گیفت‌های تلگرام خوش اومدی!',
            '',
            '🔥 این ربات با قدرت LiveProto ساخته شده و قابلیت‌های خفنی داره:',
            '',
            '⚡️ دستورات موجود:',
            '• /start - شروع کار با ربات',
            '• @me - اضافه شدن به لیست دریافت اطلاعیه',
            '• /remove - حذف از لیست',
            '• /settings - تنظیمات شخصی',
            '• /stats - آمار گیفت‌ها',
            '• /help - راهنما',
            '',
            '🎯 برای دریافت اطلاعیه گیفت‌های جدید، دستور @me رو بفرست!',
            '',
            '💎 ساخته شده با ❤️ توسط <a href="https://t.me/LiveProto">LiveProto</a>'
        ]);
        
        list($message, $entities) = $update->html($welcomeMsg);
        $update->reply(message: $message, entities: $entities);
    }
    
    #[Filter(new NewMessage(new Command(me: CommandType::AT)))]
    public function addUser($update): void {
        $peerId = $update->getPeerId();
        $username = $update->getFromUsername() ?: "کاربر_" . abs(crc32($peerId));
        
        if (isset($this->users[$peerId])) {
            list($message, $entities) = $update->html(
                '⚠️ <b>شما قبلاً در لیست هستید!</b>' . PHP_EOL .
                '🔔 در حال حاضر اطلاعیه‌های گیفت رو دریافت می‌کنید.'
            );
        } else {
            $this->users[$peerId] = [
                'username' => $username,
                'joined_at' => date('Y-m-d H:i:s'),
                'notifications' => true,
                'language' => 'fa',
                'alert_types' => ['limited', 'premium', 'rare']
            ];
            $this->saveData();
            
            list($message, $entities) = $update->html(
                '🎉 <b>عالی! به لیست اضافه شدی!</b>' . PHP_EOL .
                '🔔 از الان اطلاعیه گیفت‌های جدید رو دریافت می‌کنی' . PHP_EOL .
                '⚙️ برای تنظیمات بیشتر /settings رو بفرست'
            );
        }
        
        $update->reply(message: $message, entities: $entities);
    }
    
    #[Filter(new NewMessage(new Command(remove: [CommandType::SLASH])))]
    public function removeUser($update): void {
        $peerId = $update->getPeerId();
        
        if (isset($this->users[$peerId])) {
            unset($this->users[$peerId]);
            $this->saveData();
            
            list($message, $entities) = $update->html(
                '✅ <b>با موفقیت از لیست حذف شدید</b>' . PHP_EOL .
                '😢 دیگه اطلاعیه دریافت نمی‌کنید' . PHP_EOL .
                '🔄 برای برگشت @me رو بفرستید'
            );
        } else {
            list($message, $entities) = $update->html(
                '❌ <b>شما در لیست نیستید!</b>' . PHP_EOL .
                '💡 برای اضافه شدن @me رو بفرستید'
            );
        }
        
        $update->reply(message: $message, entities: $entities);
    }
    
    #[Filter(new NewMessage(new Command(settings: [CommandType::SLASH])))]
    public function showSettings($update): void {
        $peerId = $update->getPeerId();
        
        if (!isset($this->users[$peerId])) {
            list($message, $entities) = $update->html('❌ ابتدا با @me در ربات عضو شوید!');
            $update->reply(message: $message, entities: $entities);
            return;
        }
        
        $user = $this->users[$peerId];
        $settingsMsg = formatMessage([
            '⚙️ <b>تنظیمات شما:</b>',
            '',
            '👤 نام کاربری: %s',
            '📅 تاریخ عضویت: %s',
            '🔔 اطلاعیه‌ها: %s',
            '🌍 زبان: فارسی',
            '',
            '🎯 <b>انواع اطلاعیه:</b>',
            '• گیفت‌های محدود: %s',
            '• گیفت‌های پریمیوم: %s', 
            '• گیفت‌های نادر: %s',
            '',
            '💡 برای تغییر تنظیمات از دستورات زیر استفاده کنید:',
            '/toggle_notifications - تغییر وضعیت اطلاعیه‌ها',
            '/toggle_limited - تغییر اطلاعیه گیفت‌های محدود',
            '/toggle_premium - تغییر اطلاعیه گیفت‌های پریمیوم'
        ],
            $user['username'],
            $user['joined_at'],
            $user['notifications'] ? '✅ فعال' : '❌ غیرفعال',
            in_array('limited', $user['alert_types']) ? '✅' : '❌',
            in_array('premium', $user['alert_types']) ? '✅' : '❌',
            in_array('rare', $user['alert_types']) ? '✅' : '❌'
        );
        
        list($message, $entities) = $update->html($settingsMsg);
        $update->reply(message: $message, entities: $entities);
    }
    
    #[Filter(new NewMessage(new Command(stats: [CommandType::SLASH])))]
    public function showStats($update): void {
        $totalUsers = count($this->users);
        $activeUsers = count(array_filter($this->users, fn($user) => $user['notifications']));
        
        $statsMsg = formatMessage([
            '📊 <b>آمار ربات:</b>',
            '',
            '👥 کل کاربران: %d',
            '🔔 کاربران فعال: %d',
            '🎁 کل گیفت‌های ارسال شده: %d',
            '💎 گیفت‌های یکتا: %d',
            '📅 تاریخ: %s',
            '⏰ ساعت: %s',
            '',
            '🔥 ربات در حال کار است و گیفت‌های جدید رو رصد می‌کنه!'
        ],
            $totalUsers,
            $activeUsers,
            $this->giftStats['total_sent'],
            $this->giftStats['unique_gifts'],
            persianDate(),
            date('H:i:s')
        );
        
        list($message, $entities) = $update->html($statsMsg);
        $update->reply(message: $message, entities: $entities);
    }
    
    #[Filter(new NewMessage(new Command(help: [CommandType::SLASH])))]
    public function showHelp($update): void {
        $helpMsg = formatMessage([
            '📚 <b>راهنمای کامل ربات:</b>',
            '',
            '🎯 <b>دستورات اصلی:</b>',
            '• /start - شروع کار و خوش‌آمدگویی',
            '• @me - عضویت در سیستم اطلاع‌رسانی',
            '• /remove - لغو عضویت',
            '• /settings - مشاهده و تنظیم تنظیمات',
            '• /stats - مشاهده آمار',
            '• /help - نمایش این راهنما',
            '',
            '⚙️ <b>دستورات تنظیمات:</b>',
            '• /toggle_notifications - فعال/غیرفعال کردن اطلاعیه‌ها',
            '• /toggle_limited - گیفت‌های محدود',
            '• /toggle_premium - گیفت‌های پریمیوم',
            '',
            '🎁 <b>درباره گیفت‌ها:</b>',
            'این ربات هر ۵ ثانیه یکبار گیفت‌های جدید تلگرام رو چک می‌کنه',
            'و در صورت وجود گیفت جدید، بلافاصله بهتون اطلاع میده!',
            '',
            '💡 <b>نکات مهم:</b>',
            '• فقط گیفت‌های محدود و در دسترس نمایش داده می‌شن',
            '• قیمت‌ها بر اساس Telegram Stars محاسبه میشه',
            '• اطلاعیه‌ها فوری و آنی ارسال میشن',
            '',
            '🔥 ساخته شده با قدرت LiveProto!'
        ]);
        
        list($message, $entities) = $update->html($helpMsg);
        $update->reply(message: $message, entities: $entities);
    }
    
    #[Filter(new NewMessage(new Command(toggle_notifications: [CommandType::SLASH])))]
    public function toggleNotifications($update): void {
        $peerId = $update->getPeerId();
        if (!isset($this->users[$peerId])) {
            list($message, $entities) = $update->html('❌ ابتدا با @me در ربات عضو شوید!');
            $update->reply(message: $message, entities: $entities);
            return;
        }
        $current = $this->users[$peerId]['notifications'] ?? true;
        $this->users[$peerId]['notifications'] = !$current;
        $this->saveData();
        $state = $this->users[$peerId]['notifications'] ? '✅ فعال' : '❌ غیرفعال';
        list($message, $entities) = $update->html('🔔 وضعیت اطلاعیه‌ها: ' . $state);
        $update->reply(message: $message, entities: $entities);
    }

    #[Filter(new NewMessage(new Command(toggle_limited: [CommandType::SLASH])))]
    public function toggleLimited($update): void {
        $peerId = $update->getPeerId();
        if (!isset($this->users[$peerId])) {
            list($message, $entities) = $update->html('❌ ابتدا با @me در ربات عضو شوید!');
            $update->reply(message: $message, entities: $entities);
            return;
        }
        $types = $this->users[$peerId]['alert_types'] ?? [];
        if (in_array('limited', $types, true)) {
            $types = array_values(array_filter($types, fn($t) => $t !== 'limited'));
            $state = '❌ غیرفعال شد';
        } else {
            $types[] = 'limited';
            $state = '✅ فعال شد';
        }
        $this->users[$peerId]['alert_types'] = $types;
        $this->saveData();
        list($message, $entities) = $update->html('🎯 اطلاعیه گیفت‌های محدود: ' . $state);
        $update->reply(message: $message, entities: $entities);
    }

    #[Filter(new NewMessage(new Command(toggle_premium: [CommandType::SLASH])))]
    public function togglePremium($update): void {
        $peerId = $update->getPeerId();
        if (!isset($this->users[$peerId])) {
            list($message, $entities) = $update->html('❌ ابتدا با @me در ربات عضو شوید!');
            $update->reply(message: $message, entities: $entities);
            return;
        }
        $types = $this->users[$peerId]['alert_types'] ?? [];
        if (in_array('premium', $types, true)) {
            $types = array_values(array_filter($types, fn($t) => $t !== 'premium'));
            $state = '❌ غیرفعال شد';
        } else {
            $types[] = 'premium';
            $state = '✅ فعال شد';
        }
        $this->users[$peerId]['alert_types'] = $types;
        $this->saveData();
        list($message, $entities) = $update->html('💎 اطلاعیه گیفت‌های پریمیوم: ' . $state);
        $update->reply(message: $message, entities: $entities);
    }
    
    public function sendGiftAlert(object $gift): void {
        if (empty($this->users)) return;
        
        // جلوگیری از ارسال مکرر همان گیفت
        $giftKey = $gift->id . '_' . $gift->stars;
        if (in_array($giftKey, $this->sentGifts)) return;
        
        $this->sentGifts[] = $giftKey;
        
        // تنظیم ایموجی بر اساس نوع گیفت
        $giftIcon = '🎁';
        if ($gift->require_premium) $giftIcon = '💎';
        if ($gift->stars >= 1000) $giftIcon = '⭐️';
        if ($gift->limited) $giftIcon = '🔥';
        
        $alertMsg = formatMessage([
            '%s <b>گیفت جدید در تلگرام!</b> %s',
            '',
            '🏷 <b>نام:</b> %s',
            '🆔 <b>شناسه:</b> <code>%d</code>',
            '⭐️ <b>قیمت:</b> %s ستاره',
            '💎 <b>نیاز به پریمیوم:</b> %s',
            '♻️ <b>قابل ارتقا:</b> %s',
            '🔥 <b>محدود:</b> %s',
            '🟢 <b>موجود:</b> %s',
            '',
            '⏰ <b>زمان:</b> %s',
            '📅 <b>تاریخ:</b> %s',
            '',
            '🚀 <b>زود باش که تموم نشه!</b>'
        ],
            $giftIcon, $giftIcon,
            $gift->title ?: 'بدون نام',
            $gift->id,
            number_format($gift->stars),
            $gift->require_premium ? '✅' : '❌',
            $gift->can_upgrade ? '✅' : '❌',
            $gift->limited ? '✅ (محدود!)' : '❌',
            $gift->sold_out ? '❌ تمام شده' : '✅ موجود',
            date('H:i:s'),
            persianDate()
        );
        
        // ارسال به کاربران فعال
        $activeUsers = array_filter($this->users, function($user) use ($gift) {
            if (!$user['notifications']) return false;
            
            if ($gift->limited && !in_array('limited', $user['alert_types'])) return false;
            if ($gift->require_premium && !in_array('premium', $user['alert_types'])) return false;
            
            return true;
        });
        
        if (empty($activeUsers)) return;
        
        global $client;
        $requests = [];
        
        foreach (array_keys($activeUsers) as $peerId) {
            try {
                $requests[] = [
                    'peer' => $client->get_input_peer($peerId),
                    'message' => $alertMsg,
                    'random_id' => random_int(PHP_INT_MIN, PHP_INT_MAX),
                    'parse_mode' => 'HTML'
                ];
            } catch (Exception $e) {
                error_log("خطا در ارسال به $peerId: " . $e->getMessage());
            }
        }
        
        if (!empty($requests)) {
            try {
                $client->messages->sendMessageMultiple(...$requests, responses: false);
                
                // بروزرسانی آمار
                $this->giftStats['total_sent'] += count($requests);
                $this->giftStats['unique_gifts']++;
                $this->saveData();
                
                echo "🎁 گیفت جدید به " . count($requests) . " کاربر ارسال شد!" . PHP_EOL;
            } catch (Exception $e) {
                error_log("خطا در ارسال پیام‌ها: " . $e->getMessage());
            }
        }
    }
    
    public function sendStatusUpdate(): void {
        if (empty($this->users)) return;
        
        $hour = date('H');
        $statusMsg = formatMessage([
            '🤖 <b>گزارش وضعیت ربات</b>',
            '',
            '✅ ربات فعال و در حال کار است',
            '⏰ ساعت: %s',
            '📅 تاریخ: %s',
            '👥 کاربران فعال: %d',
            '🔍 در حال رصد گیفت‌های جدید...',
            '',
            '💡 <i>این پیام هر ساعت ارسال می‌شود</i>'
        ],
            date('H:i:s'),
            persianDate(),
            count(array_filter($this->users, fn($user) => $user['notifications']))
        );
        
        global $client;
        $activeUsers = array_filter($this->users, fn($user) => $user['notifications']);
        
        if (empty($activeUsers)) return;
        
        $requests = [];
        foreach (array_keys($activeUsers) as $peerId) {
            try {
                $requests[] = [
                    'peer' => $client->get_input_peer($peerId),
                    'message' => $statusMsg,
                    'random_id' => random_int(PHP_INT_MIN, PHP_INT_MAX)
                ];
            } catch (Exception $e) {
                error_log("خطا در ارسال وضعیت به $peerId: " . $e->getMessage());
            }
        }
        
        if (!empty($requests)) {
            try {
                $client->messages->sendMessageMultiple(...$requests, responses: false);
                echo "📊 گزارش وضعیت به " . count($requests) . " کاربر ارسال شد!" . PHP_EOL;
            } catch (Exception $e) {
                error_log("خطا در ارسال گزارش وضعیت: " . $e->getMessage());
            }
        }
    }
}

// ایجاد هندلر
$handler = new UltraGiftHandler();
$client->addHandler($handler);

// لوپ اصلی برای رصد گیفت‌ها
EventLoop::unreference(EventLoop::repeat($handler->config['check_interval'], function() use($client, $handler): void {
    static $hash = 0;
    static $lastHour = '<start>';
    static $lastCheckTime = 0;
    
    if (!$client->isAuthorized() || !$client->connected) {
        return;
    }
    
    // ارسال گزارش ساعتی
    $currentHour = date('H');
    if ($lastHour !== $currentHour) {
        $lastHour = $currentHour;
        $handler->sendStatusUpdate();
    }
    
    // چک کردن گیفت‌های جدید
    try {
        $currentTime = time();
        if ($currentTime - $lastCheckTime < 5) {
            return; // جلوگیری از چک مکرر
        }
        $lastCheckTime = $currentTime;
        
        echo "🔍 در حال چک کردن گیفت‌های جدید..." . PHP_EOL;
        
        $starGifts = $client->payments->getStarGifts(hash: $hash);
        
        if ($starGifts->getClass() === 'payments.starGifts') {
            if ($hash === 0) {
                $hash = $starGifts->hash;
                echo "🔧 هش اولیه تنظیم شد: $hash" . PHP_EOL;
            } else {
                if ($hash !== $starGifts->hash) {
                    echo "🆕 تغییر در گیفت‌ها تشخیص داده شد!" . PHP_EOL;
                    $hash = $starGifts->hash;
                    
                    foreach ($starGifts->gifts as $gift) {
                        $passesLimitedFilter = isset($handler->config['show_limited_only']) && $handler->config['show_limited_only'] ? $gift->limited : true;
                        if ($gift->getClass() === 'starGift' 
                            && !$gift->sold_out 
                            && $passesLimitedFilter) {
                            
                            echo sprintf("🎁 گیفت جدید: %s (ID: %d, قیمت: %d ستاره)" . PHP_EOL,
                                $gift->title ?: 'بدون نام', 
                                $gift->id, 
                                $gift->stars
                            );
                            
                            $handler->sendGiftAlert($gift);
                        }
                    }
                }
            }
        }
    } catch (Throwable $e) {
        error_log("خطا در رصد گیفت‌ها: " . $e->getMessage());
        echo "❌ خطا در رصد گیفت‌ها: " . $e->getMessage() . PHP_EOL;
    }
}));

// راه‌اندازی ربات
echo "🚀 ربات اطلاع‌رسانی گیفت‌های تلگرام شروع به کار کرد!" . PHP_EOL;
echo "📱 کاربران می‌توانند با @me در ربات عضو شوند" . PHP_EOL;
echo "⚙️ فاصله زمانی چک: " . $handler->config['check_interval'] . " ثانیه" . PHP_EOL;

// اگر از طریق وب (SAPI غیر CLI) صدا زده شود، اجرای بلندمدت نکن
if (php_sapi_name() !== 'cli') {
	echo "OK";
	return;
}

$client->start();

?>
