<?php

declare(strict_types=1);

// ابزار مدیریت ربات گیفت الرت

echo "🛠  ابزار مدیریت ربات گیفت الرت\n";
echo "=====================================\n\n";

// بررسی آرگومان‌های ورودی
if ($argc < 2) {
    showHelp();
    exit(1);
}

$command = $argv[1];

switch ($command) {
    case 'status':
        showStatus();
        break;
    case 'users':
        showUsers();
        break;
    case 'stats':
        showStats();
        break;
    case 'backup':
        createBackup();
        break;
    case 'restore':
        if (!isset($argv[2])) {
            echo "❌ نام فایل بک‌آپ مشخص نشده!\n";
            echo "💡 مثال: php manage.php restore backup_2024-01-01.zip\n";
            exit(1);
        }
        restoreBackup($argv[2]);
        break;
    case 'clean':
        cleanLogs();
        break;
    case 'config':
        showConfig();
        break;
    case 'reset':
        resetData();
        break;
    case 'help':
        showHelp();
        break;
    default:
        echo "❌ دستور نامعتبر: $command\n\n";
        showHelp();
        exit(1);
}

function showHelp(): void {
    echo "📚 راهنمای دستورات:\n\n";
    echo "php manage.php <command>\n\n";
    echo "دستورات موجود:\n";
    echo "  status    - نمایش وضعیت ربات\n";
    echo "  users     - مشاهده لیست کاربران\n";
    echo "  stats     - نمایش آمار کامل\n";
    echo "  backup    - ایجاد بک‌آپ\n";
    echo "  restore   - بازیابی از بک‌آپ\n";
    echo "  clean     - پاک‌سازی لاگ‌ها\n";
    echo "  config    - نمایش تنظیمات\n";
    echo "  reset     - بازنشانی داده‌ها\n";
    echo "  help      - نمایش این راهنما\n\n";
    echo "🔗 مثال‌ها:\n";
    echo "  php manage.php status\n";
    echo "  php manage.php backup\n";
    echo "  php manage.php restore backup_2024-01-01.zip\n";
}

function showStatus(): void {
    echo "📊 وضعیت ربات:\n";
    echo "================\n\n";
    
    // بررسی فایل‌های ضروری
    $files = [
        'gift_alert_bot.php' => 'فایل اصلی ربات',
        'config.php' => 'فایل تنظیمات',
        'data/users.json' => 'اطلاعات کاربران',
        'data/stats.json' => 'آمار ربات'
    ];
    
    foreach ($files as $file => $desc) {
        $status = file_exists($file) ? '✅' : '❌';
        $size = file_exists($file) ? ' (' . formatBytes(filesize($file)) . ')' : '';
        echo "  $status $desc$size\n";
    }
    
    echo "\n📁 دایرکتوری‌ها:\n";
    $dirs = ['data', 'logs', 'backup', 'temp'];
    foreach ($dirs as $dir) {
        $status = is_dir($dir) ? '✅' : '❌';
        $writable = is_writable($dir) ? 'قابل نوشتن' : 'غیرقابل نوشتن';
        echo "  $status $dir ($writable)\n";
    }
    
    // آمار سریع
    if (file_exists('data/users.json')) {
        $users = json_decode(file_get_contents('data/users.json'), true) ?: [];
        echo "\n👥 کاربران: " . count($users) . " نفر\n";
        
        $activeUsers = array_filter($users, fn($user) => $user['notifications'] ?? false);
        echo "🔔 کاربران فعال: " . count($activeUsers) . " نفر\n";
    }
    
    if (file_exists('data/stats.json')) {
        $stats = json_decode(file_get_contents('data/stats.json'), true) ?: [];
        echo "📨 کل پیام‌های ارسالی: " . ($stats['total_sent'] ?? 0) . "\n";
        echo "🎁 گیفت‌های یکتا: " . ($stats['unique_gifts'] ?? 0) . "\n";
    }
    
    // اطلاعات سیستم
    echo "\n💻 اطلاعات سیستم:\n";
    echo "  PHP: " . PHP_VERSION . "\n";
    echo "  حافظه: " . ini_get('memory_limit') . "\n";
    echo "  زمان: " . date('Y-m-d H:i:s') . "\n";
}

function showUsers(): void {
    echo "👥 لیست کاربران:\n";
    echo "==================\n\n";
    
    if (!file_exists('data/users.json')) {
        echo "❌ فایل کاربران یافت نشد!\n";
        return;
    }
    
    $users = json_decode(file_get_contents('data/users.json'), true) ?: [];
    
    if (empty($users)) {
        echo "📭 هیچ کاربری ثبت نشده!\n";
        return;
    }
    
    echo "📊 تعداد کل: " . count($users) . " کاربر\n\n";
    
    $headers = sprintf("%-15s %-20s %-20s %-10s", "ID", "نام کاربری", "تاریخ عضویت", "اطلاعیه");
    echo $headers . "\n";
    echo str_repeat("-", strlen($headers)) . "\n";
    
    foreach ($users as $userId => $user) {
        $username = $user['username'] ?? 'نامشخص';
        $joinDate = $user['joined_at'] ?? 'نامشخص';
        $notifications = ($user['notifications'] ?? false) ? '✅' : '❌';
        
        printf("%-15s %-20s %-20s %-10s\n", 
            substr($userId, 0, 14), 
            substr($username, 0, 19), 
            substr($joinDate, 0, 19), 
            $notifications
        );
    }
    
    // آمار تفصیلی
    $activeUsers = array_filter($users, fn($user) => $user['notifications'] ?? false);
    $newUsers = array_filter($users, fn($user) => 
        strtotime($user['joined_at'] ?? '1970-01-01') > strtotime('-7 days')
    );
    
    echo "\n📈 آمار:\n";
    echo "  🔔 فعال: " . count($activeUsers) . " نفر\n";
    echo "  🆕 جدید (۷ روز اخیر): " . count($newUsers) . " نفر\n";
}

function showStats(): void {
    echo "📈 آمار کامل:\n";
    echo "==============\n\n";
    
    if (!file_exists('data/stats.json')) {
        echo "❌ فایل آمار یافت نشد!\n";
        return;
    }
    
    $stats = json_decode(file_get_contents('data/stats.json'), true) ?: [];
    
    echo "📊 آمار ربات:\n";
    echo "  📨 کل پیام‌های ارسالی: " . number_format($stats['total_sent'] ?? 0) . "\n";
    echo "  🎁 گیفت‌های یکتا: " . number_format($stats['unique_gifts'] ?? 0) . "\n";
    echo "  🕐 شروع کار: " . ($stats['start_time'] ?? 'نامشخص') . "\n";
    echo "  🔄 آخرین راه‌اندازی: " . ($stats['last_restart'] ?? 'نامشخص') . "\n";
    
    // آمار کاربران
    if (file_exists('data/users.json')) {
        $users = json_decode(file_get_contents('data/users.json'), true) ?: [];
        echo "\n👥 آمار کاربران:\n";
        echo "  📊 کل کاربران: " . count($users) . " نفر\n";
        
        $activeUsers = array_filter($users, fn($user) => $user['notifications'] ?? false);
        echo "  🔔 فعال: " . count($activeUsers) . " نفر\n";
        
        $inactiveUsers = count($users) - count($activeUsers);
        echo "  🔕 غیرفعال: " . $inactiveUsers . " نفر\n";
        
        // آمار عضویت روزانه
        $joinDates = array_map(fn($user) => 
            date('Y-m-d', strtotime($user['joined_at'] ?? '1970-01-01')), 
            $users
        );
        $joinCounts = array_count_values($joinDates);
        arsort($joinCounts);
        
        echo "\n📅 عضویت در روزهای اخیر:\n";
        $i = 0;
        foreach ($joinCounts as $date => $count) {
            if ($i >= 5) break;
            echo "  $date: $count نفر\n";
            $i++;
        }
    }
    
    // آمار فایل‌ها
    echo "\n💾 اطلاعات فایل‌ها:\n";
    $files = [
        'data/users.json' => 'کاربران',
        'data/stats.json' => 'آمار',
        'logs/bot.log' => 'لاگ'
    ];
    
    foreach ($files as $file => $desc) {
        if (file_exists($file)) {
            $size = formatBytes(filesize($file));
            $modified = date('Y-m-d H:i:s', filemtime($file));
            echo "  📄 $desc: $size (آخرین تغییر: $modified)\n";
        }
    }
}

function createBackup(): void {
    echo "💾 ایجاد بک‌آپ...\n";
    
    $backupName = 'backup_' . date('Y-m-d_H-i-s') . '.zip';
    $backupPath = 'backup/' . $backupName;
    
    // ایجاد دایرکتوری بک‌آپ
    if (!is_dir('backup')) {
        mkdir('backup', 0755, true);
    }
    
    $zip = new ZipArchive();
    
    if ($zip->open($backupPath, ZipArchive::CREATE) !== TRUE) {
        echo "❌ خطا در ایجاد فایل بک‌آپ!\n";
        return;
    }
    
    // فایل‌های مورد نظر برای بک‌آپ
    $filesToBackup = [
        'gift_alert_bot.php',
        'config.php',
        'data/users.json',
        'data/stats.json',
        'data/config.json'
    ];
    
    $addedFiles = 0;
    foreach ($filesToBackup as $file) {
        if (file_exists($file)) {
            $zip->addFile($file);
            $addedFiles++;
            echo "  ✅ $file\n";
        }
    }
    
    $zip->close();
    
    if ($addedFiles > 0) {
        $size = formatBytes(filesize($backupPath));
        echo "\n✅ بک‌آپ با موفقیت ایجاد شد!\n";
        echo "📁 نام فایل: $backupName\n";
        echo "📏 حجم: $size\n";
        echo "📅 تاریخ: " . date('Y-m-d H:i:s') . "\n";
        
        // پاک‌سازی بک‌آپ‌های قدیمی (نگهداری ۷ فایل اخیر)
        cleanOldBackups();
    } else {
        unlink($backupPath);
        echo "❌ هیچ فایلی برای بک‌آپ یافت نشد!\n";
    }
}

function restoreBackup(string $backupFile): void {
    echo "🔄 بازیابی از بک‌آپ...\n";
    
    $backupPath = 'backup/' . $backupFile;
    
    if (!file_exists($backupPath)) {
        echo "❌ فایل بک‌آپ یافت نشد: $backupFile\n";
        return;
    }
    
    // تأیید کاربر
    echo "⚠️  این عملیات داده‌های فعلی را جایگزین می‌کند!\n";
    echo "آیا مطمئن هستید؟ [y/N]: ";
    $handle = fopen("php://stdin", "r");
    $line = trim(fgets($handle));
    fclose($handle);
    
    if (strtolower($line) !== 'y') {
        echo "❌ عملیات لغو شد.\n";
        return;
    }
    
    $zip = new ZipArchive();
    
    if ($zip->open($backupPath) !== TRUE) {
        echo "❌ خطا در باز کردن فایل بک‌آپ!\n";
        return;
    }
    
    $extracted = $zip->extractTo('./');
    $zip->close();
    
    if ($extracted) {
        echo "✅ بازیابی با موفقیت انجام شد!\n";
        echo "🔄 لطفاً ربات را مجدداً راه‌اندازی کنید.\n";
    } else {
        echo "❌ خطا در استخراج فایل‌ها!\n";
    }
}

function cleanLogs(): void {
    echo "🧹 پاک‌سازی لاگ‌ها...\n";
    
    $logFiles = glob('logs/*.log');
    $totalSize = 0;
    $fileCount = 0;
    
    foreach ($logFiles as $logFile) {
        $size = filesize($logFile);
        $totalSize += $size;
        
        // نگهداری ۱ مگابایت از آخر فایل
        if ($size > 1024 * 1024) {
            $content = file_get_contents($logFile);
            $newContent = substr($content, -1024 * 1024);
            file_put_contents($logFile, $newContent);
            $fileCount++;
            echo "  🗜 $logFile فشرده شد\n";
        }
    }
    
    // پاک‌سازی فایل‌های موقت
    $tempFiles = glob('temp/*');
    foreach ($tempFiles as $tempFile) {
        if (is_file($tempFile)) {
            unlink($tempFile);
            $fileCount++;
        }
    }
    
    echo "\n✅ پاک‌سازی انجام شد!\n";
    echo "📄 فایل‌های پردازش شده: $fileCount\n";
    echo "💾 فضای آزاد شده: " . formatBytes($totalSize) . "\n";
}

function showConfig(): void {
    echo "⚙️ تنظیمات ربات:\n";
    echo "==================\n\n";
    
    if (!file_exists('config.php')) {
        echo "❌ فایل تنظیمات یافت نشد!\n";
        return;
    }
    
    $config = require 'config.php';
    
    echo "🤖 اطلاعات ربات:\n";
    echo "  نام: " . $config['bot']['name'] . "\n";
    echo "  نسخه: " . $config['bot']['version'] . "\n";
    echo "  سازنده: " . $config['bot']['author'] . "\n";
    
    echo "\n📱 تنظیمات تلگرام:\n";
    echo "  API ID: " . $config['telegram']['api_id'] . "\n";
    echo "  Device Model: " . $config['telegram']['device_model'] . "\n";
    echo "  App Version: " . $config['telegram']['app_version'] . "\n";
    
    echo "\n🔍 تنظیمات رصد:\n";
    echo "  فاصله چک: " . $config['monitoring']['check_interval'] . " ثانیه\n";
    echo "  فقط محدود: " . ($config['monitoring']['show_limited_only'] ? 'بله' : 'خیر') . "\n";
    echo "  فقط موجود: " . ($config['monitoring']['show_available_only'] ? 'بله' : 'خیر') . "\n";
    
    echo "\n🔔 تنظیمات اطلاعیه:\n";
    echo "  گزارش ساعتی: " . ($config['notifications']['hourly_status'] ? 'فعال' : 'غیرفعال') . "\n";
    echo "  خلاصه روزانه: " . ($config['notifications']['daily_summary'] ? 'فعال' : 'غیرفعال') . "\n";
    echo "  حداکثر تعداد دسته‌ای: " . $config['notifications']['max_batch_size'] . "\n";
}

function resetData(): void {
    echo "🔄 بازنشانی داده‌ها...\n";
    
    echo "⚠️  این عملیات تمام داده‌ها را پاک می‌کند!\n";
    echo "آیا مطمئن هستید؟ [y/N]: ";
    $handle = fopen("php://stdin", "r");
    $line = trim(fgets($handle));
    fclose($handle);
    
    if (strtolower($line) !== 'y') {
        echo "❌ عملیات لغو شد.\n";
        return;
    }
    
    // ایجاد بک‌آپ قبل از بازنشانی
    echo "💾 ایجاد بک‌آپ امنیتی...\n";
    createBackup();
    
    // پاک کردن فایل‌ها
    $filesToReset = [
        'data/users.json',
        'data/stats.json',
        'data/config.json'
    ];
    
    foreach ($filesToReset as $file) {
        if (file_exists($file)) {
            unlink($file);
            echo "  🗑 $file حذف شد\n";
        }
    }
    
    // ایجاد فایل‌های اولیه
    file_put_contents('data/users.json', '{}');
    file_put_contents('data/stats.json', json_encode([
        'total_sent' => 0,
        'unique_gifts' => 0,
        'start_time' => date('Y-m-d H:i:s'),
        'last_restart' => date('Y-m-d H:i:s')
    ], JSON_PRETTY_PRINT));
    
    echo "\n✅ بازنشانی کامل شد!\n";
    echo "🔄 ربات آماده استفاده مجدد است.\n";
}

function cleanOldBackups(): void {
    $backups = glob('backup/backup_*.zip');
    if (count($backups) <= 7) return;
    
    // مرتب‌سازی بر اساس زمان تغییر
    usort($backups, fn($a, $b) => filemtime($b) - filemtime($a));
    
    // حذف بک‌آپ‌های اضافی
    $toDelete = array_slice($backups, 7);
    foreach ($toDelete as $backup) {
        unlink($backup);
    }
    
    if (!empty($toDelete)) {
        echo "🗑 " . count($toDelete) . " بک‌آپ قدیمی حذف شد\n";
    }
}

function formatBytes(int $bytes, int $precision = 2): string {
    $units = ['B', 'KB', 'MB', 'GB'];
    
    for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
        $bytes /= 1024;
    }
    
    return round($bytes, $precision) . ' ' . $units[$i];
}

?>
